<?php
/*--------------------------------------------------------------------
 AddOptionValuesAction.php 2021-03-25
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2021 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Api\Modules\ProductVariant\App\Actions;

use Exception;
use Gambio\Admin\Modules\ProductVariant\Services\ProductVariantsGenerationService as ProductVariantsGenerationServiceInterface;
use Gambio\Api\Modules\ProductVariant\App\ProductVariantApiRequestParser;
use Gambio\Api\Modules\ProductVariant\App\ProductVariantApiRequestValidator;
use Gambio\Core\Application\Http\Request;
use Gambio\Core\Application\Http\Response;

/**
 * Class AddOptionValuesAction
 * @package Gambio\Api\Modules\ProductVariant\App\Actions
 */
class AddOptionValuesAction
{
    /**
     * @var ProductVariantApiRequestValidator
     */
    private $validator;
    
    /**
     * @var ProductVariantsGenerationServiceInterface
     */
    private $service;
    /**
     * @var ProductVariantApiRequestParser
     */
    private $parser;
    
    
    /**
     * RemoveOptionAction constructor.
     *
     * @param ProductVariantApiRequestValidator         $validator
     * @param ProductVariantApiRequestParser            $parser
     * @param ProductVariantsGenerationServiceInterface $service
     */
    public function __construct(
        ProductVariantApiRequestValidator $validator,
        ProductVariantApiRequestParser $parser,
        ProductVariantsGenerationServiceInterface $service
    ){
        $this->validator = $validator;
        $this->parser    = $parser;
        $this->service   = $service;
    }
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        if (($productId = (int)$request->getAttribute('productId')) === 0) {
        
            return $response->withStatus(400)->withJson(['errors' => ['product id can\'t be 0']]);
        }
    
        $optionAndOptionValueIds = $request->getParsedBody();
        
        $errors = $this->validator->validateGenerationBody($optionAndOptionValueIds);
    
        if (empty($errors) === false) {
        
            return $response->withStatus(400)->withJson(['errors' => $errors]);
        }
    
        try {
    
            $parsedData = $this->parser->parseAddOptionValuesFromDocumentData($optionAndOptionValueIds);
            $variantIds = [];
            
            foreach ($parsedData as $optionId => $optionValueIds) {
                
                $result = $this->service->addOptionToExistingProductVariants($productId, $optionId, $optionValueIds);
    
                foreach ($result as $variantId) {
                    
                    $variantIds[] = $variantId->value();
                }
            }
    
            return $response->withStatus(200)->withJson(['data' => $variantIds]);
            
        } catch (Exception $exception) {
    
            return $response->withStatus(422)->withJson(['errors' => [$exception->getMessage()]]);
        }
    }
}